'use client';

import Image from 'next/image';
import { useLanguage } from '@/contexts/language-context';
import { PlaceHolderImages } from '@/lib/placeholder-images';
import { motion } from 'framer-motion';
import { Calendar, MapPin, Fuel, Users, Wrench, Beaker, Signal, Cpu, Zap, Shield, Briefcase, ArrowLeft, ArrowRight } from 'lucide-react';
import React from 'react';
import {
  Carousel,
  CarouselContent,
  CarouselItem,
  CarouselNext,
  CarouselPrevious,
} from '@/components/ui/carousel';
import Autoplay from 'embla-carousel-autoplay';
import { Card, CardContent, CardFooter } from '@/components/ui/card';
import { Button } from './ui/button';
import Link from 'next/link';

const iconMap: { [key: string]: React.ElementType } = {
  Fuel,
  Users,
  Wrench,
  Beaker,
  Signal,
  Cpu,
  Zap,
  Shield,
  Briefcase
};

const cardColors = [
  'from-blue-600/80 to-cyan-600/80',
  'from-emerald-600/80 to-green-600/80',
  'from-purple-600/80 to-violet-600/80',
  'from-amber-600/80 to-yellow-600/80',
  'from-rose-600/80 to-pink-600/80',
  'from-indigo-600/80 to-sky-600/80',
];

export function ProjectGallery() {
  const { content, language } = useLanguage();
  const { projects } = content;

  return (
    <section id="projects" className="scroll-mt-20 relative overflow-hidden bg-gradient-to-br from-slate-50 via-white to-blue-50/30 py-20 px-4">
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <motion.div
          className="absolute top-0 -left-40 w-96 h-96 bg-blue-400/10 rounded-full blur-3xl"
          animate={{ x: [0, 100, 0], y: [0, -50, 0], scale: [1, 1.2, 1] }}
          transition={{ duration: 20, repeat: Infinity, ease: "easeInOut" }}
        />
        <motion.div
          className="absolute bottom-0 -right-40 w-96 h-96 bg-purple-400/10 rounded-full blur-3xl"
          animate={{ x: [0, -100, 0], y: [0, 50, 0], scale: [1, 1.3, 1] }}
          transition={{ duration: 25, repeat: Infinity, ease: "easeInOut" }}
        />
      </div>

      <div className="relative z-10 max-w-7xl mx-auto">
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.6 }}
          className="mx-auto max-w-3xl text-center mb-16"
        >
          <div className="inline-flex items-center gap-2 px-4 py-2 bg-gradient-to-r from-blue-600/10 to-purple-600/10 rounded-full mb-6 backdrop-blur-sm border border-blue-200/50">
            <Briefcase className="w-4 h-4 text-blue-600" />
            <span className="text-sm font-semibold bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
              {projects.badge}
            </span>
          </div>
          <h2 className="font-bold tracking-tight text-slate-900 text-4xl sm:text-5xl mb-6 relative">
            {projects.title}
          </h2>
          <p className="mt-6 text-lg text-slate-600">
            {projects.subtitle}
          </p>
        </motion.div>

        <Carousel
            opts={{
              align: 'start',
              loop: true,
            }}
            plugins={[
              Autoplay({
                delay: 5000,
                stopOnInteraction: true,
              }),
            ]}
            className="w-full"
          >
            <CarouselContent className="-ml-4">
              {projects.items.map((project, index) => {
                const image = PlaceHolderImages.find((img) => img.id === project.image);
                const Icon = iconMap[project.icon] || Briefcase;

                return (
                  <CarouselItem key={index} className="pl-4 md:basis-1/2 lg:basis-1/3">
                    <motion.div 
                      className="h-full p-2"
                      initial={{ opacity: 0, y: 30 }}
                      whileInView={{ opacity: 1, y: 0 }}
                      viewport={{ once: true }}
                      transition={{ duration: 0.5, delay: index * 0.1 }}
                    >
                      <Card className="group h-full relative flex flex-col overflow-hidden rounded-2xl shadow-xl transition-all duration-500 hover:shadow-2xl hover:-translate-y-1 border-2 border-slate-200/50 hover:border-blue-300/50 bg-white/70 backdrop-blur-sm">
                        
                        {image && (
                           <CardContent className="p-0">
                              <div className="relative aspect-video rounded-t-xl overflow-hidden">
                                <Image
                                  src={image.imageUrl}
                                  alt={project.title}
                                  fill
                                  className="object-cover group-hover:scale-105 transition-transform duration-500"
                                  data-ai-hint={image.imageHint}
                                />
                                <div className="absolute inset-0 bg-gradient-to-t from-black/40 to-transparent"></div>
                                <div className={`absolute top-3 left-3 flex items-center gap-2 px-3 py-1 rounded-full bg-gradient-to-r ${cardColors[index % cardColors.length]} text-white text-xs font-bold shadow-lg`}>
                                  <Calendar className="w-3 h-3" />
                                  <span>{project.date}</span>
                                </div>
                              </div>
                           </CardContent>
                        )}
                        
                        <div className="flex flex-col flex-grow p-6">
                            <div className="flex items-center gap-2 mb-3 text-sm text-slate-500">
                                <MapPin className="w-4 h-4 text-purple-600" />
                                <span className="font-semibold">{project.location}</span>
                              </div>
                          <h3 className="text-xl font-bold text-slate-900 mb-2 flex-grow">
                            {project.title}
                          </h3>
                          <p className="text-sm text-slate-600 leading-relaxed">
                            {project.description}
                          </p>
                        </div>
                        
                        <CardFooter className="p-6 pt-0 mt-auto flex-col items-start gap-4">
                            <div className={`flex items-center gap-3 p-3 rounded-xl w-full bg-gradient-to-r ${cardColors[index % cardColors.length].replace('/80', '/20')}`}>
                                <div className={`flex h-10 w-10 items-center justify-center rounded-lg bg-gradient-to-br ${cardColors[index % cardColors.length]}`}>
                                  <Icon className="w-5 h-5 text-white" />
                                </div>
                                <span className="font-semibold text-primary text-sm">{project.icon} Project</span>
                            </div>
                             <Button asChild className="w-full group/button" variant="outline">
                                <Link href="https://www.noorenergy-iq.com/project2.html" target="_blank" rel="noopener noreferrer">
                                    {language === 'ar' ? 'المزيد من التفاصيل' : 'More Details'}
                                    <ArrowRight className="ml-2 h-4 w-4 transition-transform group-hover/button:translate-x-1" />
                                </Link>
                            </Button>
                        </CardFooter>

                      </Card>
                    </motion.div>
                  </CarouselItem>
                );
              })}
            </CarouselContent>
             <CarouselPrevious className="absolute left-[-1.5rem] top-1/2 -translate-y-1/2 bg-white/90 hover:bg-white shadow-xl border-2 border-primary/20 hover:border-accent transition-all duration-300" />
            <CarouselNext className="absolute right-[-1.5rem] top-1/2 -translate-y-1/2 bg-white/90 hover:bg-white shadow-xl border-2 border-primary/20 hover:border-accent transition-all duration-300" />
          </Carousel>
      </div>
    </section>
  );
}
